/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Windows.Forms;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.UmlRt;
using Borland.Eco.Handles;

namespace Borland.Eco.AutoContainers
{
	///<summary>
	///Enumeration determining which visibility scopes shall be shown in the autoform.
	///</summary>
	public enum AutoContainerMemberVisibility
	{
		///<summary>All members are shown.</summary>
		AllMembers,
		///<summary>Public + protected members are shown.</summary>
		ProtectedOrHigher,
		///<summary>Only public members are shown.</summary>
		PublicOnly
	}

	///<summary>
	///Enumeration determining which visibility scopes shall be shown in the autoform.
	///</summary>
	public enum ContainerReusage
	{
		///<summary>Always create new form.</summary>
		NeverReuse,
		///<summary>Reuse form on a per-class basis.</summary>
		ReuseForClass,
		///<summary>Reuse form on a per-instance basis.</summary>
		ReuseForElement
	}

	///<summary>
	///Parameter class sent into the autocontainer and AutoContainerFactory.
	///This way the signature can change without requiring changing all consumers.
	///</summary>
	public class AutoContainerArgs
	{
		private bool m_forceReadOnly;
		private EcoSpace m_EcoSpace;
		private AutoContainerMemberVisibility m_MemberVisibility;
		private ContainerReusage m_ContainerReusage;
		private bool m_ShowModal;
		///<summary>
		///Simpler constructor.
		///<see cref="ForceReadOnly"/> will be set to false,
		///<see cref="MemberVisibility"/> will be set to <see cref="AutoContainerMemberVisibility.AllMembers"/>,
		///<see cref="ContainerReusage"/> will be set to ContainerReusage.NeverReuse.
		///<see cref="ShowModal"/> will be set to false.
		///<param name="ecoSpace"> The EcoSpace the form lives in</param>
		///</summary>
		public AutoContainerArgs(EcoSpace ecoSpace)
		{
			m_MemberVisibility = AutoContainerMemberVisibility.AllMembers;
			m_EcoSpace = ecoSpace;
			m_ContainerReusage = ContainerReusage.NeverReuse;
		}
		///<summary>
		///Constructor
		///<param name="ecoSpace"> The EcoSpace the form lives in</param>
		///<param name="forceReadOnly"> indicates if the new form should be readOnly.</param>
		///<param name="memberVisibility"> indicates what level visibility of members should appear on the form.</param>
		///<param name="containerReusage"> indicates how forms shall be reused.</param>
		///<param name="showModal"> indicates if the form is intended to show modally.</param>
		///</summary>
		public AutoContainerArgs(EcoSpace ecoSpace, bool forceReadOnly, AutoContainerMemberVisibility memberVisibility, ContainerReusage containerReusage, bool showModal)
		{
			m_forceReadOnly = forceReadOnly;
			m_MemberVisibility = memberVisibility;
			m_EcoSpace = ecoSpace;
			m_ContainerReusage = containerReusage;
			m_ShowModal = showModal;
		}
		///<summary>
		///Constructor. ShowModal defaults to false.
		///<param name="ecoSpace"> The EcoSpace the form lives in</param>
		///<param name="forceReadOnly"> indicates if the new form should be readOnly.</param>
		///<param name="memberVisibility"> indicates what level visibility of members should appear on the form.</param>
		///<param name="containerReusage"> indicates how forms shall be reused.</param>
		///</summary>
		public AutoContainerArgs(EcoSpace ecoSpace, bool forceReadOnly, AutoContainerMemberVisibility memberVisibility, ContainerReusage containerReusage)
		{
			m_forceReadOnly = forceReadOnly;
			m_MemberVisibility = memberVisibility;
			m_EcoSpace = ecoSpace;
			m_ContainerReusage = containerReusage;
		}
		///<summary>Indicates if the new form should be readOnly.</summary>
		public bool ForceReadOnly { get { return m_forceReadOnly; } }
		///<summary>Indicates what level visibility of members should appear on the form.</summary>
		public AutoContainerMemberVisibility MemberVisibility { get { return m_MemberVisibility; } }
		///<summary>The EcoSpace the form lives in.</summary>
		public EcoSpace EcoSpace { get { return m_EcoSpace; } }
		///<summary>Indicates how forms shall be reused.</summary>
		public ContainerReusage ContainerReusage { get { return m_ContainerReusage; } }
		///<summary>Indicates if the form is intended to show modally.</summary>
		public bool ShowModal { get { return m_ShowModal; } }
	}
	///<summary>
	///Interface to implement to be an autocontainer.
	///An autocontainerFactory should create instances of the implementor.
	///</summary>.
	public interface IAutoContainer
	{
		/// <summary>
		/// BuildControls (optionally) creates any widgets etc to sit in the container.
		/// It's like a constructor.
		/// </summary>
		void BuildControls(IElement element, AutoContainerArgs autoContainerArgs);
		/// <summary>
		/// HookUpGUI is called after BuildControls. It hooks in the object
		/// to the handle in the container.
		/// </summary>
		/// <note>This method must be callable several times on the same container instance,
		/// as the instance may be reused.</note>
		void HookUpGUI(EcoSpace ecoSpace, IElement element);
		/// <summary>
		/// Used to set up the EcoSpace of the AutoContainer.
		/// </summary>
		EcoSpace EcoSpace { set ; }
		///<summary>
		///Allows the autocontainer to show itself, using the information in the args - maybe to show modally, 
		///change colors or what not.
		///</summary>
		void Show(AutoContainerArgs autoContainerArgs);
	}
	///<summary>
	///The service interface for AutoContainerService.
	///</summary>
	///<remarks>
	///Should probably be defined in interfaces somewhere.
	///</remarks>
	public interface IAutoContainerService
	{
		///<summary>
		///Checks if there is a an AutoContainer that can be reused and returns it. If not calls CreateContainer to return a new container.
		///</summary>
		IAutoContainer RetrieveContainer(IElement element, AutoContainerArgs autoContainerArgs);
		///<summary>
		///Registers an existing container for an element so that subsequent calls to RetrieveContainer will get this container, and not create a new
		///</summary>
		void RegisterContainerForElement(IAutoContainer container, IElement element);
		///<summary>
		///Finds a suitable factory for element and returns the container it creates.
		///</summary>
		IAutoContainer CreateContainer(IElement element, AutoContainerArgs autoContainerArgs);
		///<summary>
		///Plugs in yet another factory that can handle a set of elements.
		///<param name="autoContainerFactory">Instance of factory that will be added</param>
		///</summary>
		void AddFactory(IAutoContainerFactory autoContainerFactory);
		///<summary>
		///Removes a factory of the same type as the instance passed as parameter.
		///<param name="autoContainerFactory">Instance of factory that will be removed</param>
		///<returns>The method will return false if the removal couldn't be done (if there was no factory of the same type).</returns>
		///</summary>
		bool RemoveFactory(IAutoContainerFactory autoContainerFactory);
		///<summary>
		///Returns a clone of the list of registered factories.
		///<returns>a clone of the list of registered factories.</returns>
		///</summary>
		ArrayList RegisteredFactories();
	}
	///<summary>
	///The interface to implement to participate in handing out AutoContainers.
	///An instance of the implementing class must be regsistered on AutoContainerFactory.AddFactory().
	///</summary>
	public interface IAutoContainerFactory
	{
		/// <summary>
		/// It would be possible for the factory to return an existing container
		/// </summary>
		IAutoContainer AutoContainer { get; }
		/// <summary>
		/// Each factory determines if it handles the modelElement
		/// </summary>
		bool Matches(IModelElement modelElement);
	}
	/// <summary>
	/// "Last resort" AutoContainer implementation. Very generic. Handles all classes.
	/// </summary>
	public class AutoContainerFactory: IAutoContainerFactory
	{
		internal AutoContainerFactory() { }
		// IAutoContainerFactory
		public IAutoContainer AutoContainer { get { return new DefaultAutoForm(); } }
		// IAutoContainerFactory
		public bool Matches(IModelElement modelElement)
		{
			return modelElement is IClassifier;
		}
	}
	/// <summary>
	/// Implementation of IAutoContainerService. The actual service returned in property Instance.
	/// </summary>
	///<remarks>
	/// This should probably be hidden away as a singleton retrieved by some getService.
	///</remarks>
	public class AutoContainerService: IAutoContainerService
	{
		private readonly static AutoContainerService m_Instance = new AutoContainerService();
		///<summary>
		///Returns a singleton instance of AutoContainerService
		///</summary>
		public static IAutoContainerService Instance
		{
			get { return m_Instance; }
		}
		public AutoContainerService()
		{
			// Register default containers:
			// + Container for instances
			AddFactory(new AutoContainerFactory());
			// + Container for lists
			AddFactory(new DefaultListContainerFactory());
		}
		// IAutoFormService
		///<exception cref="ArgumentNullException">Thrown if <paramref name="element"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="autoContainerArgs"/> is null.</exception>
		public IAutoContainer CreateContainer(IElement element, AutoContainerArgs autoContainerArgs)
		{
			if (element == null) throw new ArgumentNullException("element");
			if (autoContainerArgs == null)
				throw new ArgumentNullException("autoContainerArgs");
			IAutoContainer result = AutoContainerForClassifier(element.UmlType);
			if (result != null)
			{
				result.EcoSpace = autoContainerArgs.EcoSpace;
				result.BuildControls(element, autoContainerArgs);
				result.HookUpGUI(autoContainerArgs.EcoSpace, element);
			}
			return result;
		}
		Hashtable m_HashElement = new Hashtable();
		Hashtable m_HashClass = new Hashtable();

		private void RemoveContainerFromHash(Hashtable ht, object c)
		{
			ArrayList keys = new ArrayList(ht.Count);
			foreach (object key in ht.Keys)
				keys.Add(key);
			foreach (object key in keys)
				if (ht[key].Equals(c))
					ht.Remove(key);
		}
		public void OnContainerDisposed(object sender, EventArgs e)
		{
			RemoveContainerFromHash(m_HashElement, sender);
			RemoveContainerFromHash(m_HashClass, sender);
		}

		private void StoreContainer(IAutoContainer container, IElement element)
		{
			if (!m_HashElement.ContainsKey(element))
				m_HashElement.Add(element, container);
			if (!m_HashClass.Contains(element.UmlType))
				m_HashClass.Add(element.UmlType, container);
			Control control = container as Control;
			if (control != null)
				control.Disposed += new EventHandler(OnContainerDisposed);
		}
		private static IAutoContainer RetrieveFromHashtable(Hashtable hashtable, object key)
		{
			IAutoContainer Result = (IAutoContainer)hashtable[key];
			Control control = Result as Control;
			if ((control != null) &&
					(control.Disposing || control.IsDisposed))
			{
				hashtable.Remove(key);
				Result = null;
			}
			return Result;
		}

		private IAutoContainer RetrieveContainerForClass(IClassifier classifier)
		{
			return RetrieveFromHashtable(m_HashClass, classifier);
		}
		private IAutoContainer RetrieveContainerForElement(IElement element)
		{
			return RetrieveFromHashtable(m_HashElement, element);
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="autoContainerArgs"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="element"/> is null.</exception>
		public IAutoContainer RetrieveContainer(IElement element, AutoContainerArgs autoContainerArgs)
		{
			if (autoContainerArgs == null) throw new ArgumentNullException("autoContainerArgs"); // Do not localize
			if (element == null) throw new ArgumentNullException("element"); // do not localize
			IAutoContainer Result = null;
			switch (autoContainerArgs.ContainerReusage)
			{
				case ContainerReusage.ReuseForElement:
					Result = RetrieveContainerForElement(element);
					break;

				case ContainerReusage.ReuseForClass:
					Result = RetrieveContainerForClass(element.UmlType);
					break;
			}

			if (Result == null)
			{
				Result = CreateContainer(element, autoContainerArgs);
				if (autoContainerArgs.ContainerReusage != ContainerReusage.NeverReuse)
					StoreContainer(Result, element);
			}
			else
			{
				Result.HookUpGUI(autoContainerArgs.EcoSpace, element);
			}
			return Result;
		}

		public void RegisterContainerForElement(IAutoContainer container, IElement element)
		{
			m_HashElement[element] = container;
		}                                      

		// Backwards compatibility
		public IAutoContainer CreateContainer(EcoSpace ecoSpace, IElement element)
		{
			return CreateContainer(element, new AutoContainerArgs(ecoSpace, false, AutoContainerMemberVisibility.AllMembers, ContainerReusage.NeverReuse, false));
		}

		private ArrayList mapperCollection = new ArrayList();
		// IAutoContainerService
		///<exception cref="ArgumentNullException">Thrown if <paramref name="autoContainerFactory"/> is null.</exception>
		public void AddFactory(IAutoContainerFactory autoContainerFactory)
		{
			if (autoContainerFactory == null) throw new ArgumentNullException("autoContainerFactory"); // Do not localize
			mapperCollection.Add(autoContainerFactory);
		}
		// IAutoContainerService
		///<exception cref="ArgumentNullException">Thrown if <paramref name="autoContainerFactory"/> is null.</exception>
		public bool RemoveFactory(IAutoContainerFactory autoContainerFactory)
		{
			if (autoContainerFactory == null) throw new ArgumentNullException("autoContainerFactory"); // Do not localize
			IAutoContainerFactory removeFactory = null;
			foreach (IAutoContainerFactory factory in mapperCollection)
			{
				if (factory.GetType() == autoContainerFactory.GetType())
				{
					removeFactory = factory;
					break;
				}
			}
			mapperCollection.Remove(removeFactory);
			return removeFactory != null;
		}
		// IAutoContainerService
		public ArrayList RegisteredFactories()
		{
			return (ArrayList)mapperCollection.Clone();
		}

		private IAutoContainer AutoContainerForClassifier(IClassifier classifier)
		{
			// Counting backwards means last entered wins
			for (int i = mapperCollection.Count - 1; i >= 0; i--)
			{
				if ((((IAutoContainerFactory)mapperCollection[i])).Matches(classifier))
					return ((IAutoContainerFactory)mapperCollection[i]).AutoContainer;
			}
			return null;
		}
	}
}

